unit Convert;
{
		Creates schema (.SCH) file for any comma-separated ASCII file, to allow
  importing ASCII files into database tables via the BDE.

  Original version by:		Richard Beeston
  												(posted as CSV.ZIP)
  This version by:				Mark Wilsdorf			71211.553@compuserve.com
  												Flagship Technologies, Inc.
	REFERENCE:
  *	See the file 'asciidrv.txt' in the \delphi\doc directory for a description
  	of reading ASCII file data using schema (*.SCH) files and the BDE.

	IMPROVEMENTS:
  *	Uses PChar types instead of String types, where necessary, to
  	allow reading ASCII files with lines longer than 255 characters.
	*	Correctly handles situation where a quoted string contains quotation
  	marks, i.e.:		""This is a field with nested quotes""
	*	Correctly identifies negative numeric values as Integer or Float fields
  *	9/27/96 (Ed Davis) fixed segment boundary wrap problem when first field
  	of file was empty.

  COMMENTS...
  *	ListBox display of fields sometimes works incorrectly but remains in
  	place...may be useful for debugging.
  *	Not tested with Date, Time data.
}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, DBTables, DB, Grids, DBGrids;

Const
  Delimiter = '"';
  Separator = ',';
  Special   = '';			{Alt 127 to replace Delimiter inside delimited strings}
  MaxFieldCount = 128;	{Max fields that may be encountered in ASCII file}
  MaxLine = 2048;				{Max possible length of line in ASCII file}
  MaxField = 512;				{Max possible length of a single field in ASCII file + terminating null character}
  		{Array of TFieldType names for display in ListBox}
  FieldTypeStr : Array[TFieldType] of String[15] =
    ('ftUnknown','ftString','ftSmallint','ftInteger','ftWord',
     'ftBoolean','ftFloat','ftCurrency','ftBCD','ftDate','ftTime',
     'ftDateTime','ftBytes','ftVarBytes','ftBlob','ftMemo','ftGraphic');

type
			{Storage for list of field types & length}
  FieldsType = record
    Count    : Word;							{number of fields encountered}
    Field    : Array[1..MaxFieldCount] of record
      FieldIs  : TFieldType;
      FieldLen : Byte;
      Decimals : byte;						{no. of decimal places--Float fields only}
    end;
  end;

type
  TForm1 = class(TForm)
    Table1: TTable;						{Comma-separated ASCII file}
    Table2: TTable;           {Database file to crease from Table1}
    BatchMove1: TBatchMove;
    OpenDialog1: TOpenDialog;
    Button1: TButton;
    Edit1: TEdit;
    Label1: TLabel;
    ListBox1: TListBox;
    Edit2: TEdit;
    Label2: TLabel;
    procedure Button1Click(Sender: TObject);
  private
		procedure GetFieldSizeAndType(buf: PChar; Nth: Word; var Field: TFieldType; var Size: Byte; var Decpts: Byte);
    procedure EnumerateFieldTypes(Filename : String; var Fields : FieldsType);
    procedure DefineFields(var Table2 : TTable; Filename : String);
  end;

var
  Form1: TForm1;

implementation

{$R *.DFM}

type
	Linebuftype = array[0..MaxLine] of char;
	LineBufTypePtr = ^Linebuftype;
	FieldBufType = array[0..MaxField] of char;
	FieldBufTypePtr = ^FieldBufType;
  MyException = class(Exception);
var
  Linebuf: PChar; 	{^LineBufType - buffer for reading line from ASCII file}
  Fieldbuf: PChar; 	{^FieldBufType - buffer for reading field from ASCII file}


function Word2Str(W: Word): String;
{
	Convert word value to a string
}
var
  S : String;
begin
  Str(W,S);
  Word2Str := S;
end;


procedure PreProcess(buf: PChar);
{
	Convert Separators found between Delimiter pairs into the Special character.
	i.e. "Hello world, How are you", becomes
		    "Hello world How are you"
}
var
  InText : Boolean;			{flags that we're within text quoted by Delimiter}
  c, prevc : PChar;     {ptr to char & previous char within buf}
  bufend :	PChar;			{ptr to terminating null}
begin
  c := StrScan(buf, Delimiter);
  if c<>nil then begin				{if delimiters are present...}
    InText := False;
    bufend := StrEnd(buf);
    prevc := nil;

    while (c <> bufend) do begin		{traverse entire buffer}
      if c^ = Delimiter then begin
      			{If prev char not a delim, invert flag state, else keep flag state}
            {(allows nested Delimiters)}
      	if (prevc = nil) or (prevc^ <> Delimiter) then
        	InText := not InText
        else
        			{Handle situation where a single pair of Delimiters occurs between}
              {Separators...i.e.:        ,"",     }

              {prevc is a Delimiter...so...if we're InText & next char will}
              {end the field, invert flag state}
        	if (InText) and
          	 ( ((c+1)^ = #0) or ((c+1)^ = Separator) ) then
        		InText := not InText;
			end else
      	if (c^ = Separator) and InText then
        	c^ := Special;

      prevc := c;
      Inc(c);
    end;
  end;
end;


function CountFields(buf: PChar): Word;
{
	Count fields within passed buffer (line of ASCII file)
}
var
  count: Word;				{number of fields}
  p: PChar;
begin
  count := 1;					{count first field in the file}
  p := StrScan(buf, Separator);		{find Separator char}
  while p<>nil do begin
  	Inc(p);										{pt to next char past Separator}
    Inc(count);
	  p := StrScan(p, Separator);		{find next Separator char}
  end;
  CountFields := count;
end;


procedure	GetNthField(N: Word; buf: PChar; outbuf: PChar; var strflag: boolean);
{
	Get Nth field from buf, write to outbuf.

  If field is a string field (as determined by enclosing quote marks) set
  strflag TRUE, as removal of the enclosing quotes here will make it
  impossible for caller to determine that numerals were actually part of a
  text field, or that an empty field (no characters) was enclosed in quotes.

  We chg the value of buf to point to beginning of field,
  and p to point to end of field.
}
label
	Alldone;
var
  p: PChar;
  count: Word;
begin
  count := 0;
  p := buf-1;				{point 1 ahead of buf, because we Inc() below}

  		{loop to find Nth field, leaving p pointing to next Separator or nil,}
      {and buf pointing to first character of the field}
  while (count<>N) do begin
    Inc(count);
    buf := p + 1;							{point to 1st char past Separator}
  	p := StrScan(buf, Separator);
  end;

  		{make p point to last char of field}
  if p=nil then
  	p := StrEnd(buf) - 1
	else
        {Ed Davis 9/27/96...}
        {If p=buf (after above strscan) then field is empty
        (i.e.:  , "Field2", "Field3", etc.)...and we DON'T want to Dec(p) if
        it's the first field (count=1) because that can wrap back around the
        segment boundary, making p > buf...which breaks the logic farther below,
        so just complete processing by setting outbuf^ := #0}
    if p = buf then
    	goto AllDone
    else
	    Dec(p);				{1 before next Separator}

  		{Exclude any *pair* of delimiters around the string field}
      {(allow single delimiter...assume is part of field's data)}
  if (buf^ = Delimiter) and (p^ = Delimiter) then begin
    Inc(buf);
    Dec(p);
    strflag := True;			{Alert caller that this is a string field}
	end;

  		{Copy field to outbuf}
  while (buf<=p) do begin	{MUST use '<=', because buf > p if field is empty!}
  	if buf^ = Special then
    	outbuf^ := Separator	{Convert Special chars back to Separators}
    else
  		outbuf^ := buf^;
    Inc(buf);
    Inc(outbuf);
  end;
AllDone:
	outbuf^ := #0;			{Add terminating null}
end;


procedure TForm1.GetFieldSizeAndType(buf: PChar; Nth: Word; var Field: TFieldType; var Size: Byte; var Decpts: Byte);
{
	Determine type of Nth field of passed buf, & store fieldtype in passed
  Field struct, size in passed Size, decimals (float fields) in Decpts

  Note:  At this point, we are assured that at least N fields
  exist...no need to check!
}
label cklength;
var
	c:	PChar;					{ptr into Fieldbuf global}
  bufend: PChar;			{ptr to Null past end of Fieldbuf}
  Numbs, Letts : Boolean;
  Punct : String;
  StrFlag : boolean;			{GetNthField says field is a string field}
begin
  Field := ftUnknown;
  Size := 0;
  StrFlag := False;
  		{Get Nth field into Fieldbuf global}
  GetNthField(Nth, buf, Fieldbuf, StrFlag);
	if (StrFlag = True) then begin
   	Field := ftString;
		goto cklength;
	end;

  c := Fieldbuf;
  if c^<>#0 then begin			{Determine field type}
    Numbs := False;
    Letts := False;
    Punct := '';
    bufend := StrEnd(Fieldbuf);

    while (c<>bufend) do begin
      if not Letts and (c^ in ['A'..'Z',' ','a'..'z']) then
        Letts := True;
      if not Numbs and (c^ in ['0'..'9']) then
        Numbs := True;
      if not (c^ in ['A'..'Z',' ','a'..'z','0'..'9']) then
        if Pos(c^, Punct)=0 then  	{If char not in Punct list, add it}
          Punct := Punct + c^;
			Inc(c);
    end;

    if Numbs and not Letts then begin
      if Punct='' then
        Field := ftInteger
      else begin
        		{Has Numbs & Punct, so could be...}
            {Date, Time, Float, or negative Integer}
        if Length(Punct)=1 then begin
          Case Punct[1] of
            ':' : Field := ftTime;
            '/' : Field := ftDate;
            '.' : Field := ftFloat;
            '-' : Field := ftInteger;			{neg. Integer}
          else
            	Field := ftString;
          end;{case}
        end else
        			{Has more than one Punct, could be...}
              {Float or String}
          if (Pos('-',Punct)<>0) and (Pos('.',Punct)<>0) then
            Field := ftFloat
          else
          	Field := ftString;
      end;

    end else   {Has Letts, is string}
      Field := ftString;
  end;

cklength:
  case Field of
  	ftString :
			Size := StrLen(Fieldbuf);
  	ftFloat :
    	Decpts := StrEnd(Fieldbuf) - 1 - StrScan(Fieldbuf, '.');
	end;
end;


Procedure TForm1.EnumerateFieldTypes(Filename : String; var Fields : FieldsType);
{
	Make list of the field types in Filename (ASCII file), & store list in
  Fields struct.
}
var
  Fil    		: TextFile;         {ASCII file}
  CurrCount : Word;
  TmpIs  		: TFieldType;				{temporary holder for field type}
  TmpLen 		: Byte;            	{temporary holder for field len}
  TmpDecpts : Byte;            	{temporary holder for no. of decimals in Float field}
  FileLineCount:	Longint;			{curr line counter for ASCII file}
begin
  FillChar(Fields, SizeOf(Fields), 0);
  FileLineCount := 0;

  try
    AssignFile(Fil, Filename);			{Read ASCII file Filename}
    Reset(Fil);
    with Fields do begin
      Count := 0;										{field count}
      ListBox1.Clear;
      repeat
        Readln(Fil, LineBufTypePtr(Linebuf)^);       {Read next line of ASCII file}
        Inc(FileLineCount);
        Edit2.SetTextBuf(Linebuf);		{display it}
        Edit2.Refresh;
        PreProcess(Linebuf);					{Cvrt separators within delims to special char}
        CurrCount := CountFields(Linebuf);
        if (CurrCount<>Count) then begin  {Verify currcount same as Fields.Count for each line}
          if (Count<>0) then
            raise MyException.Create('Inconsistant number of fields in ASCII file!')
          else begin
            Count := CurrCount;
            Edit1.Text := Word2Str(CurrCount);       {Display field count}
            Edit1.Refresh;
          end;
        end;

        		{Fill 'CurrCount' field's element of FieldsType array}
        for CurrCount := 1 to Count do begin
        	with Field[CurrCount] do begin
            GetFieldSizeAndType(Linebuf, CurrCount, TmpIs, TmpLen, TmpDecpts);

                {Is curr field's type == stored field type?}
            if (TmpIs<>FieldIs) then begin
              if FieldIs=ftUnknown then begin
                ListBox1.Items.Add(Word2Str(CurrCount)+' '+FieldTypeStr[TmpIs]+' '+Word2Str(TmpLen));
                ListBox1.Refresh;
              end else if TmpIs<>ftUnknown then begin
                raise MyException.Create('Field '+Chr(CurrCount+ Ord('0'))+#39+'s type changed in line:'
                						+ IntToStr(FileLineCount) + '!');
              end;
            end;

                {If this field's type not=default, then store it}
            if TmpIs<>ftUnknown then begin
              FieldIs := TmpIs;
                  {If field's length has increased, store & show it}
              if TmpLen>FieldLen then begin
                FieldLen := TmpLen;
                ListBox1.Items.Add(Word2Str(CurrCount)+' '+FieldTypeStr[FieldIs]+' '+Word2Str(TmpLen));
                ListBox1.Items.Exchange(CurrCount-1, ListBox1.Items.Count-1);
                ListBox1.Items.Delete(ListBox1.Items.Count-1);
                ListBox1.Refresh;
              end;
                  {If (Float) field's decimal place count has increased, store it}
            if (TmpIs=ftFloat) and (TmpDecpts > Decimals) then
              	Decimals := TmpDecpts;
            end;
          end;
        end;
      until Eof(Fil);		{read ENTIRE file--assures we measure max len of strings encountered}

      for CurrCount := 1 to Count do begin
      	with Field[CurrCount] do begin
              {Any fields we cannot understand are strings!}
          if FieldIs=ftUnknown then
            FieldIs := ftString;
              {Any string fields with no length must be at least 1 in length!}
          if (FieldLen=0) and (FieldIs=ftString) then
            FieldLen := 1;
        end;
      end;
    end;
  finally
    CloseFile(Fil);
  end;
end;


procedure TForm1.DefineFields(var Table2: TTable; Filename: String);
{
	Create .SCH for ASCII file & setup database Table2 to hold the ASCII records
}
var
  Fields : FieldsType;
  Fil    : TextFile;
  FldCount	: Word;
begin
			{Get list of all field types in Filename (ASCII), store in Fields}
  EnumerateFieldTypes(Filename, Fields);
			{Clear Table2's field defns & indexes}
  with Table2 do begin
    FieldDefs.Clear;
    IndexDefs.Clear;
  end;
			{Write schema file}
  AssignFile(Fil, Copy(Filename,1,Pos('.',Filename)-1) + '.SCH');
  ReWrite(Fil);
  Writeln(Fil, '[', ExtractFilename(Copy(Filename,4,Pos('.',Filename)-4)) ,']');
  Writeln(Fil,'Filetype=VARYING');
  Writeln(Fil,'Delimiter=' + Delimiter);
  Writeln(Fil,'Separator=' + Separator);
  Writeln(Fil,'CharSet=ascii');
  with Fields do begin
    for FldCount := 1 to Count do with Field[FldCount] do begin
      Write(Fil,'Field',FldCount,'=','Field',FldCount,',');		 {contrived field name}
      Case FieldIs of
        ftInteger : Writeln(Fil, 'LONGINT,20,0,0');
				ftFloat   : Writeln(Fil, 'FLOAT,20,', Decimals, ',0');
        ftDate    : Writeln(Fil, 'DATE,', FieldLen, ',0,0');
        ftTime    : Writeln(Fil, 'TIME,', FieldLen, ',0,0');
      else
        Writeln(Fil, 'CHAR,', FieldLen, ',0,0');
      end;
      		{Define field names in Table2 for ASCII file's fields}
          {NOTE:  FieldLen = 0 for Float, Integer fields}
      Table2.FieldDefs.Add('Field' + Word2Str(FldCount), FieldIs, FieldLen, False);
      		{Index Table2 on first field}
      {NOTE:  Following causes key violation during BatchMove if first fields}
      {				of any records in ASCII file are not unique. Uncomment the }
      {				following lines if you wish to index.}
      {if FldCount=1 then}
      {  Table2.IndexDefs.Add('Field' + Word2Str(FldCount) + 'Index',}
      {  			'Field' + Word2Str(FldCount), [ixPrimary, ixUnique]);}
    end;
  end;
  CloseFile(Fil);
  Table2.CreateTable;
end;


procedure TForm1.Button1Click(Sender: TObject);
var
  Define : Boolean;
  DBFile : String;
  Button : Integer;
  P      : Byte;
begin
  if OpenDialog1.Execute then begin

    Table1.Tablename := OpenDialog1.Filename;   {Table1 is the ASCII file}
    P := Pos('.',OpenDialog1.Filename);
    DBFile := Copy(OpenDialog1.Filename,1,P-1); {get file name, not extension}
    Define := True;
    Button := IDNO;           									{WinAPI: No button selected}

    		{Prep to create DB table of same name as ASCII fileopen name}
    with Table2 do begin
      Active := False;
      Databasename := DBFile;
      TableName := DBFile + '.DB';
      TableType := ttParadox;
    end;

    if FileExists(DBFile + '.DB') then begin
      Button := Application.MessageBox('Delete Old Table and Continue?',
      						'Table Exists', mb_YesNoCancel + mb_DefButton1);
      Define := (Button = IDYES);
    end;

    if Define then begin
			New(LineBufTypePtr(Linebuf));			{Alloc unit global buffers}
			New(FieldBufTypePtr(Fieldbuf));
    			{Define .SCH file, & Field names in Table2}
      DefineFields(Table2, OpenDialog1.Filename);

			Dispose(Fieldbuf);
			Dispose(Linebuf);
		end;

    if Button<>IDCANCEL then
      BatchMove1.Execute;
  end;
end;

end.
